#' scrdb pipeline
#'
#' pipeline for reading mars-seq, clustering.  Plotting is done by sc_pipe_plots
#'

#' sc_pipe_clean
#'
#' reading mars-seq and cleaning the umi table
#'
#' @param param_fn 
#' @param index_fn 
#' @param base_dir 
#' @param batch_meta_attr 
#' @param mars_batches 
#' @param sample_n_batches 
#' @param outdir 
#' @param outdir_add_timestamp 
#' @param clust_knn 
#' @param min_umi_n 
#' @param min_umis_init 
#' @param mark.min_var_mean 
#' @param mark.sz_cor_norm_max 
#' @param mark.niche_T 
#' @param mark.min_tot 
#' @param glob_blacklist_terms 
#' @param mark_blacklist_terms 
#' @param outlier_gene_top1 
#' @param outlier_gene_on_count 
#' @param remove_outliers_before_clustering 
#' @param min_clust_size 
#' @param filt_amb_on_clusts 
#' @param filt_outliers_on_clusts 
#' @param store_scmat_fn_pref 
#' @param amb_epsilon 
#'
#' @return returns invisibly a tgScMat
#'
#' @export
sc_pipe_clean = function(param_fn = NA,
									 index_fn = "batch_index.txt",
									 base_dir = ".",
									 batch_meta_attr = "amp.batch",
									 mars_batches = NULL,
									 sample_n_batches = NA,
									 outdir = ".",
									 outdir_add_timestamp = F,
									 clust_knn = 150,
									 min_umi_n = 100,
									 max_umi_n=10000,
									 min_umis_init = min(200, min_umi_n),
									 mark.min_var_mean = 0.2,
									 mark.sz_cor_norm_max = -Inf,
									 mark.niche_T = Inf,
									 mark.min_tot = NA,
									 glob_blacklist_terms = c(),
									 mark_blacklist_terms = c(),
									 outlier_gene_top1 = 20,
									 outlier_gene_on_count = 200,
									 remove_outliers_before_clustering = F,
									 min_clust_size = 20,
									 filt_amb_on_clusts = F,
									 filt_outliers_on_clusts = F,
									 amb_epsilon = 0.03,
									 store_scmat_fn_pref = NA,
									 batches_umis_stats_fn = NULL)
{
	if (!is.na(param_fn)) {
		source(param_fn, local = T)
	}
	
	if (outdir_add_timestamp) {
		outdir = paste0(outdir, "/", format(Sys.time(), "%Y.%m.%d_%H.%M"))
	}
	
	if (!dir.exists(outdir)) {
		dir.create(outdir)
	}
	
	# read batch index
	index = read.delim(index_fn, stringsAsFactors = F, header = T)
	
	# use selected batches
	if (!is.null(mars_batches)) {
		bat = intersect(mars_batches, index[, batch_meta_attr])
		index = index[index[, batch_meta_attr] %in% bat,]
	}
	
	# sample batches
	if (!is.na(sample_n_batches)) {
		sample_n_batches = min(sample_n_batches, nrow(index))
		index = index[sample(1:nrow(index), sample_n_batches, replace = F),]
	}
	# read all mars UMIs
	sc_mat = scm_read_scmat_mars(
		base_dir = base_dir,
		mars_batches = index[, batch_meta_attr],
		batch_meta = index,
		min_umis_n = min_umis_init,
		max_umis_n=max_umi_n,
		stats_fn_pref = batches_umis_stats_fn,
	)
	
	sc_mat_orig = sc_mat

	if (length(glob_blacklist_terms) > 0) {
		# blacklist for markers
		glob_blist = c()
		allg = rownames(sc_mat@mat)
		for (bad in glob_blacklist_terms) {
			glob_blist = union(glob_blist, grep(bad, allg, v = T))
		}
		if (length(glob_blist) > 0) {
			sc_mat = scm_sub_mat(sc_mat,
													 genes = setdiff(rownames(sc_mat@mat), glob_blist))
		}
	}
	
	# remove cross well contamination
	sc_mat = scm_remove_ambient_by_epsilon(sc_mat, amb_epsilon, batch_meta_attr, min_umi_n = min_umi_n)
	# calc gstat
	
	
	# remove_outliers (extract to scmat?)
	if (remove_outliers_before_clustering) {
		gstat = umi_gene_stat(sc_mat)
		outlier_genes = rownames(gstat[gstat$ds_top1 > outlier_gene_top1
																	 &
																	 	gstat$is_on_count < outlier_gene_on_count,])
		if (length(outlier_genes) > 1) {
			outlier_cells = names(which(colSums(as.matrix(sc_mat@mat)[outlier_genes,]) >
																		outlier_gene_top1))
			message(
				"removing ",
				length(outlier_cells),
				" outlier cells and ",
				length(outlier_genes),
				" outlier genes"
			)
			if (length(outlier_cells) > 0) {
				sc_mat = scm_sub_mat(
					sc_mat,
					cells = setdiff(colnames(sc_mat@mat), outlier_cells),
					genes = setdiff(rownames(sc_mat@mat), outlier_genes)
				)
			}
			gstat = gstat[setdiff(rownames(gstat), outlier_genes),]
		}
	}
	
	if (filt_amb_on_clusts | filt_outliers_on_clusts) {
		# find markers
		gstat = umi_gene_stat(sc_mat)
		marks = select_markers(
			sc_mat,
			gstat = gstat,
			type = "any",
			mark.min_var_mean = mark.min_var_mean,
			mark.sz_cor_norm_max = mark.sz_cor_norm_max,
			mark.niche_T = mark.niche_T,
			mark.min_tot = mark.min_tot
		)
		
		# blacklist for markers
		blist = c()
		allg = rownames(gstat)
		for (bad in mark_blacklist_terms) {
			blist = union(blist, grep(bad, allg, v = T))
		}
		
		marks = setdiff(marks, blist)
		if (length(marks) < 10) {
			stop("found ", length(marks), " markers only, consider changing config to allow more genes qualifying as informative")
		}
		
		plot_markers(
			mat = sc_mat@mat,
			marks =  marks,
			gstat = gstat,
			fname = "markers.png",
			outdir = outdir
		)
		
		
		# cluster as an additional tool for cleaning
		
		sc_cl = scc_cluster_knn_graph(sc_mat,
																	sc_mat@mat[marks,],
																	k_knn = clust_knn,
																	min_clust_size = min_clust_size)
	
		if (filt_amb_on_clusts) {
			deamb_mat = .scc_clean_ambient_on_clusts(sc_cl, sc_mat_orig@mat[sc_mat@genes,], amb_epsilon, batch_meta_attr)
			sc_mat = tgScMat(deamb_mat, "umi", cell_metadata = sc_mat@cell_metadata[colnames(deamb_mat),])
		}

		if (filt_outliers_on_clusts) {
			cell_outliers_nms = find_clustering_outliers(sc_cl)
			message("remove ", length(cell_outliers_nms), " outlier cells")
			sc_mat = scm_sub_mat(sc_mat, cells = setdiff(colnames(sc_mat@mat), cell_outliers_nms))
			reclust = T
		}
	

		if (!is.na(store_scmat_fn_pref)) {
			message("saving scmat tables")
			
			write.table(as.matrix(sc_mat@mat), file = sprintf("%s/%s_sc_mat.txt", outdir, store_scmat_fn_pref), quote=F, sep="\t")
			write.table(sc_mat@cell_metadata, file = sprintf("%s/%s_sc_mat_md.txt", outdir, store_scmat_fn_pref), quote=F, sep="\t")
		}
	}
	
	invisible(sc_mat)
}


#' sc_pipe_init_sort
#' 
#' Project sc_mat on given cell type markers, cluster projected mat. Should be followed by manual
#' assignment of cell types to cluster centers
#'
#' @param sc_mat 
#' @param markers_fn table with columns: subtype,gene,type
#' @param n_mark_clusts 
#'
#' @return kmeans object of the projected mat
#' @export
#'
#' @examples
sc_pipe_init_sort = function(sc_mat, markers_fn, alg_type="knn", min_clust_size=20, n_mark_clusts = 80, ordered_subtypes = c("ab_T_cell", "regulatory_CD4_T_cell", "CD4_T_cells", "CD8_T_cells", "gd_T_cell", "NK_cell", "cytotoxic_B_cells", "IgA_B_cell", "IgM_B_cell", "IgG_B_cell", "plasma_cells", "DC", "macrophage", "granulocytes", "erythrocyte", "osteoclast"), sc_cl=NULL)
{
	sc_marker_clusts(
		sc_mat,
		markers_fn = markers_fn,
		n_mark_clusts = n_mark_clusts,
		alg_type=alg_type,
		min_clust_size = min_clust_size,
		analysis_dir = getwd(),
		ordered_subtypes = ordered_subtypes,
		totmark_cl=sc_cl
	)
	
}

#' sc_pipe_sort
#'
#' Assign cells to types by finding the nearest center per cell
#' @param sc_mat
#' @param sort_types_markers_fn
#' @param sort_ref_centers_pref_fn
#' @param out_base_dir
#' @param out_base_dir_add_timestamp
#' @param sc_mat_tables_fn_pref
#' @param param_fn 
#'
#' @return returns invisibly a the original tgScMat with assigned types in @cell_metadata$assignd_type
#' @export
#'
#' @examples
sc_pipe_sort = function(sc_mat,
												sort_types_markers_fn,
												sort_ref_centers_pref_fn,
												out_base_dir,
												out_base_dir_add_timestamp = F,
												sc_mat_tables_fn_pref = NULL,
												param_fn = NA
												)
{
	
	if (!is.na(param_fn)) {
		source(param_fn, local = T)
	}
	
	if (out_base_dir_add_timestamp) {
		out_base_dir = paste0(out_base_dir, "/", format(Sys.time(), "%Y.%m.%d_%H.%M"))
	}
	
	if (!dir.exists(out_base_dir)) {
		dir.create(out_base_dir)
	}
	
	stopifnot(!is.null(sc_mat) | !is.null(sc_mat_tables_fn_pref))
	
	if (is.null(sc_mat)) {
		sc_mat = tgScMat(
			read.table(
				sprintf("%s_sc_mat.txt", sc_mat_tables_fn_pref),
				header = T,
				stringsAsFactors = F
			),
			"umi",
			cell_metadata = read.table(
				sprintf("%s_sc_mat_md.txt", sc_mat_tables_fn_pref),
				header = T,
				stringsAsFactors = F
			)
		)
	}
	
	# in-silico sort to types by reference type markers and cluster centers
	message(
		sprintf(
			"in-silico sorting cells based on cell type markers: %s and pre-defined clusters %s",
			sort_types_markers_fn,
			sort_ref_centers_pref_fn
		)
	)
	
	sc_mat = sc_marker_split(
		sc_mat,
		sort_types_markers_fn,
		sprintf("%s.txt", sort_ref_centers_pref_fn),
		sprintf("%s_assign.txt", sort_ref_centers_pref_fn),
		fig_pref = sprintf("%s/sort", out_base_dir),
		out_base_dir
	)
	
	invisible(sc_mat)
}

#' sc_pipe_cluster
#'
#' cluster input sc_mat
#'
#' @param sc_mat 
#' @param param_fn 
#' @param outdir 
#' @param outdir_add_timestamp 
#' @param clust_knn 
#' @param mark.min_var_mean 
#' @param mark.sz_cor_norm_max 
#' @param mark.niche_T 
#' @param mark.min_tot 
#' @param mark_blacklist_terms 
#' @param filt_outliers_on_clusts 
#' @param min_clust_size 
#' @param store_rda_fn 
#' @param tab_clust_fp_fn 
#' @param sc_mat_tables_fn_pref 
#' @param clust_fp_metadata_fields 
#'
#' @return invisible tgScMatClust clustering object
#' @export
#'
#' @examples
sc_pipe_cluster = function(sc_mat,
													 param_fn = NA,
													 outdir = ".",
													 outdir_add_timestamp = F,
													 clust_knn = 150,
													 mark.min_var_mean = 0.2,
													 mark.sz_cor_norm_max = -Inf,
													 mark.niche_T = Inf,
													 mark.min_tot = NA,
													 mark_blacklist_terms = c(),
													 filt_outliers_on_clusts = F,
													 min_clust_size = 30,
													 store_rda_fn = NA,
													 tab_clust_fp_fn = "clust_fp.txt",
													 clust_fp_metadata_fields = NA,
													 sc_mat_tables_fn_pref = NULL)
{
	if (!is.na(param_fn)) {
		source(param_fn, local = T)
	}
	
	if (outdir_add_timestamp) {
		outdir = paste0(outdir, "/", format(Sys.time(), "%Y.%m.%d_%H.%M"))
	}
	
	if (!dir.exists(outdir)) {
		dir.create(outdir)
	}
	
	stopifnot(!is.null(sc_mat) | !is.null(sc_mat_tables_fn_pref))
	
	if (is.null(sc_mat)) {
		message(sprintf("reading sc_mat from %s_sc_mat.txt and %s_sc_mat_md.txt", sc_mat_tables_fn_pref, sc_mat_tables_fn_pref))
		sc_mat = tgScMat(
			read.table(
				sprintf("%s_sc_mat.txt", sc_mat_tables_fn_pref),
				header = T, sep = "\t",
				stringsAsFactors = F
			),
			"umi",
			cell_metadata = read.table(
				sprintf("%s_sc_mat_md.txt", sc_mat_tables_fn_pref),
				header = T, sep = "\t",
				stringsAsFactors = F
			)
		)
	}
	
	gstat = umi_gene_stat(sc_mat)
	
	# markers
	marks = select_markers(
		sc_mat,
		gstat = gstat,
		type = "any",
		mark.min_var_mean = mark.min_var_mean,
		mark.sz_cor_norm_max = mark.sz_cor_norm_max,
		mark.niche_T = mark.niche_T,
		mark.min_tot = mark.min_tot
	)
	
	# blacklist for markers
	blist = c()
	allg = rownames(gstat)
	for (bad in mark_blacklist_terms) {
		blist = union(blist, grep(bad, allg, v = T))
	}
	
	marks = setdiff(marks, blist)
	if (length(marks) < 10) {
		stop(
			"found ",
			length(marks),
			" markers only, consider changing config to allow more genes qualifying as informative"
		)
	}
	plot_markers(
		mat = sc_mat@mat,
		marks =  marks,
		gstat = gstat,
		fname = "markers.png",
		outdir = outdir
	)
	
	# cluster
	sc_cl = scc_cluster_knn_graph(sc_mat,
																sc_mat@mat[marks, ],
																k_knn = clust_knn,
																min_clust_size = min_clust_size)
	
	reclust = F
	cell_outliers_nms = c()

	if (filt_outliers_on_clusts) {
		cell_outliers_nms = find_clustering_outliers(sc_cl)
		message("remove ", length(cell_outliers_nms), " outlier cells")
		sc_mat = scm_sub_mat(sc_mat, cells = setdiff(colnames(sc_mat@mat), cell_outliers_nms))
		reclust = T
	}
	sc_cl_1st = sc_cl
	if (reclust) {
		sc_cl = scc_cluster_knn_graph(sc_mat,
																	sc_mat@mat[marks, ],
																	k_knn = clust_knn,
																	min_clust_size = min_clust_size)
		
	}
	
	if (!is.na(tab_clust_fp_fn)) {
		f = apply(sc_cl@clust_fp, 1, max) > 1.5
		
		if (length(clust_fp_metadata_fields) > 1 || !is.na(clust_fp_metadata_fields)) {
			for (s in clust_fp_metadata_fields) {
				write.table(table(sc_cl@scmat@cell_metadata[names(sc_cl@clusts), s], sc_cl@clusts), paste0(outdir, "/", tab_clust_fp_fn, ".", s), sep = "\t", quote = F)	
				}
		}
		write.table(
			round(sc_cl@clust_fp[f, ], 2),
			paste0(outdir, "/", tab_clust_fp_fn),
			sep = "\t",
			quote = F
		)
	}
	
	if (!is.na(store_rda_fn)) {
		message("saving Rda")
		save(sc_cl,
				 sc_cl_1st,
				 cell_outliers_nms,
				 file = paste0(outdir, "/", store_rda_fn))
	}
	
	
	invisible(sc_cl)
}

#' sc_pipe_plots
#'
#' @return returns invisibly a tgScMatClust2D plotting object.
#'
#' @export
sc_pipe_plots = function(sc_cl,
	outdir = ".",
	param_fn = NA,
	mark_blacklist_terms = c(),
	mark_colors_fn = NA,
	fig2d_force_comp = F,
	fig_width_confu = 1200,
	fig_width_mat = 2000,
	fig_mat_per_clust_genes=10,
	fig_mat_gene_min_fold=2.5,
	fig_mat_gene_min_cov=0.5,
	fig_mat_width = 2000,
	fig_mat_height  = 3500,
	fig_mat_text_cex=0.8,
	fig_mat_smooth_n=10,
	fig_2d_height=1500,
	fig_2d_width = 1500,
	K_2dproj = 15,
	K_cells_2dproj = 15,
	force_max_deg_2dproj = 4,
	T_edge_2dproj=0.05,
	T_edge_asym=F,
	restrict_edges_by_fpcor=T,
	expand_K_connect_2dproj=10,
	fold_shades = colorRampPalette(rev(RColorBrewer::brewer.pal(11,"RdBu")))(1000), #(c("black", "blue", "lightblue", "white", "orange", "red", "yellow"))(1000),
	store_rda_fn = NA,
	focus_tfs_fn = NA,
	plot_genes_2d = F,
	meta_field_nm = NA,
	highlight_cells_by_meta_field_nms = NA,
	cell_type_markers_fn = NA
) {
	if(!is.na(param_fn)) {
		source(param_fn, local=T)

	}
	
	blist = c()
	allg = sc_cl@scmat@genes
	for (bad in mark_blacklist_terms) {
		blist = union(blist, grep(bad, allg, v = T))
	}
	##########
	#  plot  #
	##########
	sc_2d = scp_compute_clust_knn_graph(scp_init_plot(sc_cl),
					force_max_deg = force_max_deg_2dproj,
					K = K_2dproj,
					T_edge = T_edge_2dproj,
					T_edge_asym = T_edge_asym,
					subselect_by_fpcor=restrict_edges_by_fpcor,
					k_expand_inout_factor=expand_K_connect_2dproj)

	if(!is.na(mark_colors_fn)) {
		mcol = read.table(mark_colors_fn, sep="\t", h=T, stringsAsFactors=F)
		marker_colors = data.frame(color=mcol$color)
		rownames(marker_colors) = mcol$gene
		marker_colors$group = mcol$group
		if ("priority" %in% colnames(mcol)) {
			marker_colors$priority = mcol$priority
		}
		else {
			marker_colors$priority = 1
		}
		sc_2d = scp_set_clust_cols(sc_2d, marker_colors = marker_colors)
	}
	
	mat_genes = list(marks=NULL)
	sc_2d = scp_plot_confusion_mat(sc_2d,
																 outdir = outdir,
																 height = fig_width_confu,
																 width = fig_width_confu)
	
	if (!is.na(cell_type_markers_fn)) {
		ct_marks = read.table(cell_type_markers_fn, header=T, stringsAsFactors = F)
		mat_genes[['sort_marks']] = unique(ct_marks$gene)
	}
	
	for (s in names(mat_genes)) {
		sc_2d = scp_plot_mat(
			sc_2d,
			png_fn = sprintf("%s/%s_all_mat.png", outdir, s),
			per_clust_genes = fig_mat_per_clust_genes,
			gene_min_fold = fig_mat_gene_min_fold,
			gene_min_cov = fig_mat_gene_min_cov,
			width = fig_mat_width,
			height = fig_mat_height,
			text_cex = fig_mat_text_cex,
			smooth_n = fig_mat_smooth_n,
			blacklist = blist,
			genes = mat_genes[[s]]
		)
		sc_2d = scp_plot_mat(
			sc_2d,
			plot_cells = F,
			png_fn = sprintf("%s/%s_clust_mat.png", outdir, s),
			fp_shades = fold_shades,
			per_clust_genes = fig_mat_per_clust_genes,
			gene_min_fold = fig_mat_gene_min_fold,
			gene_min_cov = fig_mat_gene_min_cov,
			width = fig_mat_width,
			height = fig_mat_height,
			text_cex = fig_mat_text_cex,
			blacklist = blist,
			genes = mat_genes[[s]]
		)
	}
	
	# plot metadata
	if (is.na(meta_field_nm)) {
		meta_field_nm = colnames(sc_2d@scl@scmat@cell_metadata)[1]
	}
	sc_2d = scp_plot_metadata_factor(
		sc_2d,
		clust_breakdown = T,
		heatmap = T,
		meta_field_nm = meta_field_nm,
		width = fig_mat_width,
		height = fig_mat_width / 2,
		fname = paste0(outdir, "/metadata.png")
	)
	
	# projections
	sc_2d = scp_plot_clust_2d(
		sc_2d,
		fname = paste0(outdir, "/all_2d.png"),
		height = fig_2d_height,
		width = fig_2d_width,
		plot_markers = F, 
		K_for_cells = K_cells_2dproj
	)
	sc_2d = scp_plot_clust_2d(
		sc_2d,
		fname = paste0(outdir, "/graph_2d.png"),
		height = fig_2d_height,
		width = fig_2d_width,
		plot_edges = T,
		plot_markers = F,
		K_for_cells = K_cells_2dproj
	)
	
	write.table(cbind(data.frame(id=names(sc_2d@scl@clusts), x=sc_2d@x, y=sc_2d@y, clust=sc_2d@scl@clusts), sc_2d@scl@scmat@cell_metadata), paste0(outdir, "/cells_xy_metadata.txt"), sep="\t", quote=F)
	
	# color cells by metadata fields
	if (length(highlight_cells_by_meta_field_nms) > 1 || !is.na(highlight_cells_by_meta_field_nms)) {
		for (meta_field in highlight_cells_by_meta_field_nms) {
			sc_2d = scp_plot_clust_2d_by_meta_field(
				sc_2d,
				meta_field,
				fname = sprintf("%s/all_2d_by_%s.png", outdir, meta_field),
				K_for_cells = K_cells_2dproj,
				bg_col = "darkgrey",
				fg_col = "red",
				panel_size = 250,
				cex = 0.5
			)
		}
	}
	
	# TF plots
	if (!is.na(focus_tfs_fn)) {
		tfs = read.table(focus_tfs_fn, h = T)
		tfs = rownames(tfs)
		sc_2d = scp_plot_mat(
			sc_2d,
			png_fn = paste0(outdir, "/tf_all_mat.png"),
			genes_pool = tfs,
			per_clust_genes = fig_mat_per_clust_genes,
			gene_min_fold = fig_mat_gene_min_fold,
			gene_min_cov = fig_mat_gene_min_cov,
			width = fig_mat_width,
			height = fig_mat_height,
			text_cex = fig_mat_text_cex,
			smooth_n = fig_mat_smooth_n,
			blacklist = blist
		)
		
		sc_2d = scp_plot_mat(
			sc_2d,
			plot_cells = F,
			genes_pool = tfs,
			png_fn = paste0(outdir, "/tf_clust_mat.png"),
			fp_shades = fold_shades,
			per_clust_genes = fig_mat_per_clust_genes,
			gene_min_fold = fig_mat_gene_min_fold,
			gene_min_cov = fig_mat_gene_min_cov,
			width = fig_mat_width,
			height = fig_mat_height,
			text_cex = fig_mat_text_cex,
			blacklist = blist
		)
		
		cl_ord = sc_2d@cl_ord
		
		covgenes = names(rowSums(sc_cl@scmat@mat) > 80)
		tfs = intersect(tfs, covgenes)
		tfs = intersect(tfs, rownames(sc_cl@clust_fp))
		mask_folds = sc_cl@clust_fp[tfs,] * (sc_cl@clust_gcov[tfs,] > 0.5)
		spec_tfs = names(which(apply(mask_folds, 1, max) > 2))
		
		tf_cor = cor(log2(t(sc_cl@clust_fp[spec_tfs,])))
		
		tf_hc = hclust(dist(tf_cor), "ward.D2")
		tf_max_at = apply(sc_cl@clust_fp[spec_tfs, cl_ord], 1, which.max)
		tf_hc = as.hclust(reorder(as.dendrogram(tf_hc), tf_max_at, agglo.FUN =
																mean))
		png(paste0(outdir, "/tfs_hclust.png"),
				w = 1500,
				h = 1500)
		plot(as.phylo(tf_hc), type = "fan")
		dev.off()
		png(paste0(outdir, "/tfs_cor.png"),
				w = 2000,
				h = 2000)
		diag(tf_cor) = NA
		tfs_shades = colorRampPalette(
			c(
				"blue",
				"blue",
				"blue",
				"lightblue",
				"white",
				"red",
				"orange",
				"yellow",
				"black"
			)
		)(1000)
		
		image(
			tf_cor[tf_hc$order, tf_hc$order],
			col = tfs_shades,
			zlim = c(-1, 1),
			xaxt = 'n',
			yaxt = 'n'
		)
		mtext(
			spec_tfs[tf_hc$order],
			at = seq(0, 1, l = length(spec_tfs)),
			las = 2,
			side = 2,
			cex = 0.8
		)
		mtext(
			spec_tfs[tf_hc$order],
			at = seq(0, 1, l = length(spec_tfs)),
			las = 2,
			side = 4,
			cex = 0.8
		)
		mtext(
			spec_tfs[tf_hc$order],
			at = seq(0, 1, l = length(spec_tfs)),
			las = 2,
			side = 1,
			cex = 0.8
		)
		dev.off()
	}
	
	if (plot_genes_2d) {
		message("plotting genes 2d")
		if (!exists("spec_tfs")) {
			spec_tfs = rownames(sc_cl@feat_mat)
			if (!is.null(marker_colors)) {
				spec_tfs = unique(c(spec_tfs, rownames(marker_colors)))
			}
		}
		dir.create(paste0(outdir, "/genes/"))
		for (nm in spec_tfs) {
			cat("plotting ", nm, "\n")
			scp_plot_gene_2d(
				sc_2d,
				gene_nm = nm,
				base_dir = paste0(outdir, "/genes/"),
				w = 1200,
				h = 1200,
				reg_factor = 10
			)
		}
	}
	
	if (!is.na(store_rda_fn)) {
		message("saving Rda")
		save(sc_2d,  file = paste0(outdir, "/", store_rda_fn))
	}
	invisible(sc_2d)
	
}
